function [out,fmin,parout]=cosd_formd(Ffun,Ffun_sgrad,Gfun,Gfun_sgrad,startx,feas_eps,maxiter,ballflag,center,lowb,upb,ecoflag,printflag,printcon,stuckiter)
% COSD_FORMD employs the co-subgradient descent method inside the md software. 
% It is used when the underlying set is a box or a ball.
% ====================================================
% Usage:
% out               = COSD_FORMD(Ffun,Ffun_sgrad,Gfun,Gfun_sgrad,startx,feas_eps,maxiter,ballflag,center,lowb,upb,ecoflag,printflag,printcon,stuckiter)
% [out,fmin]        = COSD_FORMD(Ffun,Ffun_sgrad,Gfun,Gfun_sgrad,startx,feas_eps,maxiter,ballflag,center,lowb,upb,ecoflag,printflag,printcon,stuckiter)
% [out,fmin,parout] = COSD_FORMD(Ffun,Ffun_sgrad,Gfun,Gfun_sgrad,startx,feas_eps,maxiter,ballflag,center,lowb,upb,ecoflag,printflag,printcon,stuckiter)
% ====================================================
% Input:
% Ffun        - function handle for the objective function f
% Ffun_sgrad  - function handle for the subgradient of the objective function f
% Gfun        - function handle for the constraint function g
% Gfun_sgrad  - function handle for the subgradients of the components of the constraint function (arranged in columns)
% startx      - a starting vector
% feas_eps    - feasibility tolerance
% maxiter     - maximal number of iterations
% ballflag    - a flag indicating whether the underlying set is a ball or a box
%             - true when it is a ball, false when it is a box
% center  - center of ball (same dimensions as startx) [default: zeros(size(startx))]
%             - relevant when the underlying set is a ball 
% lowb        - lower bound on the variables
%             - relevant when the underlying set is a box 
% upb         - upper bound on the variables when the underlying set is a box
%             - radius of the ball when the underlying set is a ball 
% ecoflag     - true if economic version (without calculating objective function
%                              values) should run, otherwise false
% printflag   - true if internal printing should take place, otherwise false
% printcon    - true if constraint value should be printed, otherwise false
% stuckiter   - maximal allowed number of iterations without improvement
% ====================================================
% Output:
% out    - optimal solution (up to a tolerance)
% fmin   - optimal value (up to a tolerance)
% parout - a struct containing additional information related to the convergence. The fields of parout are:
%    iterNum   - number of performed iterations
%    funValVec - vector of all function values generated by the method
%    consVec   - vector containing all the constraints violations quantities (when relevant)

% This file is part of the FOM package - a collection of first order methods for solving convex optimization problems
% Copyright (C) 2017 Amir and Nili Beck
% 
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
% 
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
% 
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.


%starting values
startfmin = inf ;
iternu=0; %iteration number
if(ballflag)
    theta = 2*upb^2 ;
else
    if (isscalar (upb) && (isscalar(lowb)))
        [size1,size2] = size(startx) ;
        theta = 0.5*(size1*size2*(upb - lowb)^2) ; 
    else
        theta = 0.5*(norm(upb-lowb,'fro')^2) ;
    end
end


newx = startx ;
out = startx ; %best x so far

derf = Ffun_sgrad(newx) ;
[newg,derg] = ConsGetOne(Gfun(newx),Gfun_sgrad(newx),newx) ;


fmin = startfmin ;
savediter=-1; % to indicate that we haven't found any feasible solution so far
%starting parout, when necessary
if (nargout  == 3)
    parout = struct ;
    parout.iterNum = 0 ;
    parout.funValVec = [] ;
    if (printcon)
        parout.consVec  = [] ;
    end
end

%the iterations
while( (iternu < maxiter))
    iternu=iternu+1;
    
    if (newg >feas_eps)
        avec = derg ; %subgradient for objective function
    else
        avec = derf ; %subgradient for constraint function
    end
    
    gama = sqrt(theta)/sqrt(iternu);
    newx=newx - gama*avec/(norm(avec,'fro'));
    
    if (ballflag)
        newx =  proj_Euclidean_ball(newx,center,upb) ;
    else
        newx = proj_box(newx,lowb,upb) ;
    end
    
    derf = Ffun_sgrad(newx) ;
    [newg,derg] = ConsGetOne(Gfun(newx),Gfun_sgrad(newx),newx);
    
    if (~ecoflag)
        if (iternu - savediter > stuckiter)
            fprintf('Stopping because of %d iterations with no improvement\n',stuckiter) ;
            return
        end
        newf = Ffun(newx) ;
    
        if (nargout  == 3)
            %updating parout
            parout.funValVec = [ parout.funValVec ; newf] ;
            if (printcon)
                parout.consVec = [ parout.consVec ; newg] ;
            end
        end
        
        if ( (newf < fmin ) && (newg < feas_eps)  )
            fmin = newf ;
            out = newx ;
            if (printflag)
                if (printcon)
                    fprintf('%6d \t    %12f\t %12f\n',iternu,fmin,newg);
                else
                    fprintf('%6d \t    %12f \n',iternu,fmin);
                end
            end
            savediter=iternu ;
        end
    else
        if (newg < feas_eps)
            out = newx ;
            savediter=iternu ;
        end
    end
        
end

if (savediter == -1)
  %never set a value to savediter, no feasible solution found
    fprintf('No feasible solution found\n') ;
    out = zeros(size(startx)) ;
    fmin = startfmin ;
end

if (nargout == 3)
    %updating parout.iterNum
    parout.iterNum = iternu ;
end

    function [val,der]=  ConsGetOne(fun_val,fun_sgrad,x)
        %this function handles the case when the constraint function returns
        %a vector or a matrix, because there is more than one constraint
        if (isscalar (fun_val))
            val = fun_val ;
            der = fun_sgrad ;
        else %val is a vector, there is more than one constraint
            [val,pl] = max(fun_val) ;
            num_col = size(x,2) ;
            der = fun_sgrad(:,(pl-1)*num_col+1:pl*num_col) ;
        end
    end

end

